import os
import time
import json
import traceback
import FreeSimpleGUI as sg
import numpy as np
import matplotlib.pyplot as plt

from serial import Serial, SerialException
from socket import socket, AF_INET, SOCK_DGRAM
from threading import Lock
from serial.tools import list_ports
from datetime import datetime
from ast import literal_eval


READ_TIMEOUT = 5.0  # [s]

# Axis encoding used for the Gold Drives to identify the axis
AXIS_ENCODING_X = 1
AXIS_ENCODING_Y = 2

# Titanium Drives use axis prefixes 'AX<number>'
# AXIS_NUMBER_X = 2  # Initial tunings used AX2 for the X-axis
# AXIS_NUMBER_Y = 1  # Initial tunings used AX1 for the Y-axis
AXIS_NUMBER_X = 1
AXIS_NUMBER_Y = 2

# Absolut Encoder:
ENCODER_SCALING = 1 / 1000000   # [nm] -> [mm]
# Incremental Encoder:
#ENCODER_SCALING = 1 / 16384000   # LIP 608, interpolation: 14, [cnt] -> [mm]

MOTOR_IDLE = 0
MOTOR_SETTLING = 1
MOTOR_MOVING = 2
MOTOR_DISABLED = 3

MOTION_TIMEOUT = 45.0  # [s]
STOP_ON_TIMEOUT = True

SCAN_TYPES = ['Well Meander X', 'Well Meander Y', 'Well Meander Only X', 'Well Meander Only Y']

# Absolut Encoder:
ORIGIN_OFFSET_X =  101.0    # from Settings.cfg
ORIGIN_OFFSET_Y = -199.0    # from Settings.cfg
# Incremental Encoder:
#ORIGIN_OFFSET_X = 10.0
#ORIGIN_OFFSET_Y = 10.0
LOWER_LIMIT_X = ORIGIN_OFFSET_X +   2.0  # from Settings.cfg
UPPER_LIMIT_X = ORIGIN_OFFSET_X + 375.0  # from Settings.cfg
LOWER_LIMIT_Y = ORIGIN_OFFSET_Y +  -3.0  # from Settings.cfg
UPPER_LIMIT_Y = ORIGIN_OFFSET_Y +  80.0  # Settings.cfg at 86.0 is not correct and would crash

# Aproximation of 384 Revvity CellCarrier Plate
PLATE_ORIGIN_X  = 114.5     # PlateOriginX = StartX + PlateOffsetX from Settings.cfg and Plate.xml
PLATE_ORIGIN_Y  =  70.5     # PlateOriginY = StartY + PlateOffsetY from Settings.cfg and Plate.xml
PLATE_COLUMNS = 24          # from Plate.xml
PLATE_ROWS    = 16          # from Plate.xml
PLATE_WELL_X = -4.5         # Well to Well distance and direction in X
PLATE_WELL_Y = -4.5         # Well to Well distance and direction in Y
PLATE_MID_X = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + ((PLATE_COLUMNS - 1) / 2 * PLATE_WELL_X)
PLATE_MID_Y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + ((PLATE_ROWS - 1) / 2 * PLATE_WELL_Y)
PLATE_FIRST_COLUMN_X = PLATE_ORIGIN_X + ORIGIN_OFFSET_X
PLATE_LAST_COLUMN_X  = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + ((PLATE_COLUMNS - 1) * PLATE_WELL_X)
PLATE_FIRST_ROW_Y    = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y
PLATE_LAST_ROW_Y     = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + ((PLATE_ROWS - 1) * PLATE_WELL_Y)

AXIS_PARAMETERS = {
    'Default v2': {
        'SPEED_X'      :  300,  # [mm/s]
        'ACCEL_X'      :  800,  # [mm/s^2]
        'DECEL_X'      :  800,  # [mm/s^2]
        'STOP_DECEL_X' : 1000,  # [mm/s^2]

        'SPEED_Y'      :  100,  # [mm/s]
        'ACCEL_Y'      :  800,  # [mm/s^2]
        'DECEL_Y'      :  800,  # [mm/s^2]
        'STOP_DECEL_Y' : 1000,  # [mm/s^2]
    },
    'Production Test v2': {
        'SPEED_X'      :  360,  # [mm/s]
        'ACCEL_X'      :  960,  # [mm/s^2]
        'DECEL_X'      :  960,  # [mm/s^2]
        'STOP_DECEL_X' : 1000,  # [mm/s^2]

        'SPEED_Y'      :  120,  # [mm/s]
        'ACCEL_Y'      :  960,  # [mm/s^2]
        'DECEL_Y'      :  960,  # [mm/s^2]
        'STOP_DECEL_Y' : 1000,  # [mm/s^2]
    },
    'Default': {
        'SPEED_X'      :  300,  # [mm/s]
        'ACCEL_X'      :  800,  # [mm/s^2]
        'DECEL_X'      :  500,  # [mm/s^2]
        'STOP_DECEL_X' : 1000,  # [mm/s^2]

        'SPEED_Y'      :  100,  # [mm/s]
        'ACCEL_Y'      :  800,  # [mm/s^2]
        'DECEL_Y'      :  500,  # [mm/s^2]
        'STOP_DECEL_Y' : 1000,  # [mm/s^2]
    },
    'Production Test': {
        'SPEED_X'      :  360,  # [mm/s]
        'ACCEL_X'      :  960,  # [mm/s^2]
        'DECEL_X'      :  600,  # [mm/s^2]
        'STOP_DECEL_X' : 1000,  # [mm/s^2]

        'SPEED_Y'      :  120,  # [mm/s]
        'ACCEL_Y'      :  960,  # [mm/s^2]
        'DECEL_Y'      :  600,  # [mm/s^2]
        'STOP_DECEL_Y' : 1000,  # [mm/s^2]
    },
    'Slow': {
        'SPEED_X'      :   10,  # [mm/s]
        'ACCEL_X'      :  800,  # [mm/s^2]
        'DECEL_X'      :  800,  # [mm/s^2]
        'STOP_DECEL_X' : 1000,  # [mm/s^2]

        'SPEED_Y'      :   10,  # [mm/s]
        'ACCEL_Y'      :  800,  # [mm/s^2]
        'DECEL_Y'      :  800,  # [mm/s^2]
        'STOP_DECEL_Y' : 1000,  # [mm/s^2]
    },
}

FINAL_SPEED = 0
SMOOTHING_FACTOR = 20

PLATE_SCAN_RESULTS_DIR = os.path.dirname(os.path.abspath(__file__))
PLATE_SCAN_RESULTS_LOG = os.path.join(PLATE_SCAN_RESULTS_DIR, 'plate_scan_results.log')

FRICTION_SCAN_RESULTS_DIR = os.path.dirname(os.path.abspath(__file__))

PLATE_SCAN_CMAP_MIN = 100
PLATE_SCAN_CMAP_MAX = 500

MOTOR_FORCE_CONSTANT_X = 5.5  # [N/A]
MOTOR_FORCE_CONSTANT_Y = 8.1  # [N/A]


class ElmoDrive:
    IP_ADDRESS = ('192.168.2.2', 5001)

    def __init__(self, name: str, interface: Serial | socket):
        self.name = name
        self.interface = interface
        self.lock = Lock()

    def send_command(self, command: str, log=True) -> str:
        with self.lock:
            command_bytes = f"{command};".encode()
            if isinstance(self.interface, Serial):
                self.interface.write(command_bytes)
                echo = self.interface.read_until(b';').decode()
                response = self.interface.read_until(b';').decode()
            elif isinstance(self.interface, socket):
                self.interface.sendto(command_bytes, self.IP_ADDRESS)
                data, server = self.interface.recvfrom(1024)
                echo, response = data.decode().split(';', maxsplit=1)
            else:
                raise TypeError("interface must be a Serial or socket instance")
            if log:
                print_output(f"{self.name}: {echo} -> {response}")
            return response[:-1]


class ElmoAxis:
    def __init__(self, drive: ElmoDrive, number: int = 0):
        self.drive = drive
        self.number = number

    def send_command(self, command: str, log=True) -> str:
        if self.number > 0:
            command = f"AX{self.number}.{command}"
        return self.drive.send_command(command, log=log)


class ElmoAxisError(Exception):
    pass


class WindowClosed(Exception):
    pass


main_window = None

task_in_progress = False

elmo_drive_list: list[ElmoDrive] = []
elmo_axis_x: ElmoAxis = None
elmo_axis_y: ElmoAxis = None

last_position_x = None
last_position_y = None

axis_parameter = None


def main():
    global main_window
    global task_in_progress
    global elmo_axis_x
    global elmo_axis_y
    global last_position_x
    global last_position_y
    global axis_parameter

    try:
        main_layout = [
            [sg.Text('Console:', font=(*sg.DEFAULT_FONT, 'bold'))],
            [sg.Multiline(key='-OUTPUT-', size=(45,12), expand_x=True, font='Courier 10', write_only=True, disabled=True)],
            [sg.ProgressBar(100, size=(10, 10), expand_x=True, key='-PROGRESS-')],
            [sg.Text('Direct Drive Control:', font=(*sg.DEFAULT_FONT, 'bold'))],
            [
                sg.Combo([], key='-DRIVE-', font='Courier 10', readonly=True),
                sg.Input(key='-COMMAND-', size=15, expand_x=True, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Button('Send', key='-SEND-', size=5, bind_return_key=True),
            ],
            [sg.HorizontalSeparator()],
            [
                sg.Text('Scan Table Control:', font=(*sg.DEFAULT_FONT, 'bold')),
                sg.Push(),
                sg.Text('X', font=(*sg.DEFAULT_FONT, 'bold'), pad=0), sg.Text('⚫', key='-XON-', text_color='red3', pad=((0, 5), 0)),
                sg.Text('Y', font=(*sg.DEFAULT_FONT, 'bold'), pad=0), sg.Text('⚫', key='-YON-', text_color='red3', pad=((0, 10), 0)),
                sg.Button('STOP', key='-STOP-', size=5, font=(*sg.DEFAULT_FONT, 'bold'), button_color='red'),
            ],
            [
                sg.Text('Speed'), sg.Combo([], key='-SPEED-', expand_x=True, font='Courier 10', readonly=True),
                sg.Button('Initialize', key='-INITIALIZE-'),
                sg.Button('Commutation', key='-COMMUTATION-'),
                sg.Button('Download', key='-DOWNLOAD-', tooltip='Download Textual Parameters File to Drive'),
            ],
            [
                sg.Text('Current Position [mm]', size=16),
                sg.Text('X:'), sg.Input(key='-GETPOSX-', size=11, font='Courier 10', disabled_readonly_background_color='gray75', readonly=True),
                sg.Text('Y:'), sg.Input(key='-GETPOSY-', size=11, font='Courier 10', disabled_readonly_background_color='gray75', readonly=True),
            ],
            [
                sg.Text('Target Position [mm]', size=16),
                sg.Text('X:'), sg.Input(f"{PLATE_MID_X:.2f}", key='-SETPOSX-', size=11, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Text('Y:'), sg.Input(f"{PLATE_MID_Y:.2f}", key='-SETPOSY-', size=11, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Button('Move', key='-MOVE-', size=5),
            ],
            [
                sg.Text('Step Distance [mm]'),
                sg.Input('10.0', key='-STEP-', size=8, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.VerticalSeparator(),
                sg.Button('X-', key='-STEPXNEG-', font='Courier 10'),
                sg.Button('X+', key='-STEPXPOS-', font='Courier 10'),
                sg.VerticalSeparator(),
                sg.Button('Y-', key='-STEPYNEG-', font='Courier 10'),
                sg.Button('Y+', key='-STEPYPOS-', font='Courier 10'),
                sg.VerticalSeparator(),
            ],
            [sg.HorizontalSeparator()],
            [sg.Text('Plate Scan:', font=(*sg.DEFAULT_FONT, 'bold'))],
            [
                sg.Text('Type'),
                sg.Combo(SCAN_TYPES, default_value=SCAN_TYPES[0], key='-PSCANTYPE-', font='Courier 10', readonly=True),
                sg.Text('Iterations'),
                sg.Input('1', key='-PSCANITER-', size=3, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Text('Save As'),
                sg.Input(key='-PSCANSAVEAS-', size=8, font='Courier 10', disabled_readonly_background_color='gray75',
                         tooltip='Prefix for the plate scan file names in the format "<prefix>_plate_scan_YYYYMMDD_HHMMSS"'),
            ],
            [
                sg.Button('Run', key='-PSCANRUN-', size=6),
                sg.Button('Load', key='-PSCANLOAD-', size=6),
                sg.Button('Log', key='-PSCANLOG-', size=6),
            ],
            [sg.HorizontalSeparator()],
            [sg.Text('Production Test:', font=(*sg.DEFAULT_FONT, 'bold'))],
            [
                sg.Text('Iterations'),
                sg.Input('1', key='-PRODTESTITER-', size=3, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Button('Run', key='-PRODTESTRUN-', size=6),
            ],
            [sg.HorizontalSeparator()],
            [sg.Text('Friction Scan:', font=(*sg.DEFAULT_FONT, 'bold'))],
            [
                sg.Text('Fixed Pos X'), sg.Input(f"{PLATE_MID_X:.2f}", key='-FIXEDPOSX-', size=7, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Text('Fixed Pos Y'), sg.Input(f"{PLATE_MID_Y:.2f}", key='-FIXEDPOSY-', size=7, font='Courier 10', disabled_readonly_background_color='gray75'),
                sg.Text('Save As'),
                sg.Input(key='-FSCANSAVEAS-', size=8, font='Courier 10', disabled_readonly_background_color='gray75',
                         tooltip='Prefix for the friction scan file names in the format "<prefix>_friction_scan_YYYYMMDD_HHMMSS"'),
            ],
            [
                sg.Button('Run', key='-FSCANRUN-', size=6),
                sg.Button('Load', key='-FSCANLOAD-', size=6),
            ],
            [sg.HorizontalSeparator()],
            [
                sg.Button('Close', key='-CLOSE-', size=5),
            ],
        ]
        main_window = sg.Window(f"Elmo Scan Table Control", main_layout, finalize=True)

        # Check for Elmo drive via ethernet. If available ethernet will be used for x and y axis.
        udp_socket = socket(AF_INET, SOCK_DGRAM)
        try:
            udp_socket.settimeout(1.0)
            elmo_drive = ElmoDrive('ETH', udp_socket)
            firmware_version = elmo_drive.send_command('VR', log=False)
            serial_number = elmo_drive.send_command('SN[4]', log=False)
            udp_socket.settimeout(READ_TIMEOUT)

            print_output(f"Found Elmo drive:")
            print_output(f"    Interface: ETH")
            print_output(f"    Firmware: {firmware_version}")
            print_output(f"    S/N: {serial_number}")

            elmo_drive_list.append(elmo_drive)

            if firmware_version.startswith('Castanet'):
                # Dual axis controller -> Use axis number
                print_output(f"    AX{AXIS_NUMBER_X}: X-axis")
                elmo_axis_x = ElmoAxis(elmo_drive, AXIS_NUMBER_X)
                print_output(f"    AX{AXIS_NUMBER_Y}: Y-axis")
                elmo_axis_y = ElmoAxis(elmo_drive, AXIS_NUMBER_Y)

        except TimeoutError:
            udp_socket.close()

        if elmo_axis_x is None or elmo_axis_y is None:
            for port in list_ports.comports():
                if 'ELMO' in port.description:
                    try:
                        serial_port = Serial(port.device, timeout=READ_TIMEOUT)
                        elmo_drive = ElmoDrive(port.name, serial_port)
                        firmware_version = elmo_drive.send_command('VR', log=False)
                        serial_number = elmo_drive.send_command('SN[4]', log=False)

                        print_output(f"Found Elmo drive:")
                        print_output(f"    Interface: {port.name}")
                        print_output(f"    Firmware: {firmware_version}")
                        print_output(f"    S/N: {serial_number}")

                        elmo_drive_list.append(elmo_drive)

                        if firmware_version.startswith('Castanet'):
                            # Dual axis controller -> Use axis prefix
                            print_output(f"    AX{AXIS_NUMBER_X}: X-axis")
                            elmo_axis_x = ElmoAxis(elmo_drive, AXIS_NUMBER_X)
                            print_output(f"    AX{AXIS_NUMBER_Y}: Y-axis")
                            elmo_axis_y = ElmoAxis(elmo_drive, AXIS_NUMBER_Y)
                        else:
                            # Signel axis controller -> Use axis encoding and no prefix
                            axis_encoding = int(elmo_drive.send_command('UI[2]', log=False))
                            if axis_encoding == AXIS_ENCODING_X:
                                print_output(f"    Axis Encoding: X-axis")
                                elmo_axis_x = ElmoAxis(elmo_drive)
                            elif axis_encoding == AXIS_ENCODING_Y:
                                print_output(f"    Axis Encoding: Y-axis")
                                elmo_axis_y = ElmoAxis(elmo_drive)
                            else:
                                print_output(f"    Axis Encoding: unknown")

                    except SerialException as e:
                        print_output(f"Failed to connect to Elmo drive {port.name}: {repr(e)}", colors='orange')

        if elmo_drive_list == []:
            print_output(f"No Elmo drives found", colors='red')

        main_window['-DRIVE-'].update(values=[elmo_drive.name for elmo_drive in elmo_drive_list], set_to_index=0)
        main_window['-SPEED-'].update(values=list(AXIS_PARAMETERS.keys()), set_to_index=0)

        while True:

            if not task_in_progress:
                update_motor_on()
                current_position_x, current_position_y = update_position()

            event, values = main_window.read(timeout=200)

            if event == sg.WIN_CLOSED or event == '-CLOSE-':
                raise WindowClosed()

            if event == '-SEND-':
                elmo_drive = next((drive for drive in elmo_drive_list if drive.name == values['-DRIVE-']), None)
                if elmo_drive is not None:
                    elmo_drive.send_command(values['-COMMAND-'])
                else:
                    print_output(f"Selected drive not found", colors='red')

            if event == '-STOP-':
                stop()

            if event == '-INITIALIZE-':
                axis_parameter = AXIS_PARAMETERS[values['-SPEED-']]
                initialize()

            if event == '-COMMUTATION-':
                start_task(commutation)

            if event == '-DOWNLOAD-':
                start_task(download_parameters_file)

            if event == '-MOVE-':
                if values['-SETPOSX-'] != '' and values['-SETPOSY-'] == '':
                    start_task(move_x, float(values['-SETPOSX-']))
                elif values['-SETPOSX-'] == '' and values['-SETPOSY-'] != '':
                    start_task(move_y, float(values['-SETPOSY-']))
                elif values['-SETPOSX-'] != '' and values['-SETPOSY-'] != '':
                    start_task(move, float(values['-SETPOSX-']), float(values['-SETPOSY-']))

            if event == '-STEPXPOS-':
                position_x = last_position_x if last_position_x is not None else current_position_x
                start_task(move_x, position_x + float(values['-STEP-']))

            if event == '-STEPXNEG-':
                position_x = last_position_x if last_position_x is not None else current_position_x
                start_task(move_x, position_x - float(values['-STEP-']))

            if event == '-STEPYPOS-':
                position_y = last_position_y if last_position_y is not None else current_position_y
                start_task(move_y, position_y + float(values['-STEP-']))

            if event == '-STEPYNEG-':
                position_y = last_position_y if last_position_y is not None else current_position_y
                start_task(move_y, position_y - float(values['-STEP-']))

            if event == '-PSCANRUN-':
                start_task(run_plate_scan, values['-PSCANTYPE-'], int(values['-PSCANITER-']), values['-PSCANSAVEAS-'])

            if event == '-PSCANRESULT-':
                plate_scan_results, file_name = values['-PSCANRESULT-']
                show_plate_scan_results_map(plate_scan_results, file_name=file_name)
                print_plate_scan_results_summary(plate_scan_results)

            if event == '-PSCANLOAD-':
                load_plate_scan_results()

            if event == '-PSCANLOG-':
                show_plate_scan_results_log()

            if event == '-PRODTESTRUN-':
                start_task(run_production_test, int(values['-PRODTESTITER-']))

            if event == '-FSCANRUN-':
                start_task(run_friction_scan, float(values['-FIXEDPOSX-']), float(values['-FIXEDPOSY-']), values['-FSCANSAVEAS-'])

            if event == '-FSCANRESULT-':
                friction_scan_results, file_name = values['-FSCANRESULT-']
                show_friction_scan(friction_scan_results, label_list=[], file_name=file_name)

            if event == '-FSCANLOAD-':
                load_friction_scans()

            if event =='-TASK DONE-':
                task_in_progress = False

            if not task_in_progress:
                update_progress(100)
                disable_controls(False)

    except WindowClosed:
        main_window.close()

    except Exception as e:
        sg.easy_print(traceback.format_exc(), keep_on_top=True, wait=True)

    finally:
        for elmo_drive in elmo_drive_list:
            elmo_drive.interface.close()


def start_task(function, *args, **kwargs):
    global task_in_progress
    task_in_progress = True
    disable_controls(True)
    main_window.perform_long_operation(lambda: try_task(function, *args, **kwargs), '-TASK DONE-')


def try_task(function, *args, **kwargs):
    try:
        function(*args, **kwargs)
    except ElmoAxisError as e:
        print_output(f"{e}", colors='red')
    except Exception as e:
        print_output(traceback.format_exc(), colors='red')


def print_output(message, colors=None):
    main_window['-OUTPUT-'].print(message, colors=colors)
    main_window.refresh()


def disable_controls(disabled=True, ignore=[]):
    control_keys = [
        '-DRIVE-', '-COMMAND-', '-SEND-',
        '-SPEED-', '-INITIALIZE-', '-COMMUTATION-', '-DOWNLOAD-',
        '-SETPOSX-', '-SETPOSY-', '-MOVE-',
        '-STEP-', '-STEPXPOS-', '-STEPXNEG-', '-STEPYPOS-', '-STEPYNEG-',
        '-PSCANTYPE-', '-PSCANITER-', '-PSCANSAVEAS-',
        '-PSCANRUN-', '-PSCANLOAD-', '-PSCANLOG-',
        '-PRODTESTITER-', '-PRODTESTRUN-',
        '-FIXEDPOSX-', '-FIXEDPOSY-', '-FSCANRUN-', '-FSCANLOAD-', '-FSCANSAVEAS-',
    ]
    control_keys = (key for key in control_keys if key not in ignore)
    for key in control_keys:
        main_window[key].update(disabled=disabled)
    main_window.refresh()


def update_progress(percent):
    main_window['-PROGRESS-'].update(current_count=int(percent))
    main_window.refresh()


def update_position():
    position_x = 0.0
    position_y = 0.0

    if elmo_axis_x is not None:
        position_x = float(elmo_axis_x.send_command('PU', log=False)) * ENCODER_SCALING
        main_window['-GETPOSX-'].update(value=f"{position_x:11.6f}")
        main_window.refresh()
    if elmo_axis_y is not None:
        position_y = float(elmo_axis_y.send_command('PU', log=False)) * ENCODER_SCALING
        main_window['-GETPOSY-'].update(value=f"{position_y:11.6f}")
        main_window.refresh()

    return position_x, position_y


def update_motor_on():
    x_on = False
    y_on = False

    if elmo_axis_x is not None:
        x_on = elmo_axis_x.send_command('MO', log=False) == '1'
    if elmo_axis_y is not None:
        y_on = elmo_axis_y.send_command('MO', log=False) == '1'

    main_window['-XON-'].update(text_color=('green3' if x_on else 'red3'))
    main_window['-YON-'].update(text_color=('green3' if y_on else 'red3'))
    main_window.refresh()


def stop():
    if elmo_axis_x is not None:
        error = elmo_axis_x.send_command('MO=0')
        if not error:
            print_output(f"Successfully disabled X-axis", colors='green')
        else:
            print_output(f"Failed to disable X-axis", colors='red')

    if elmo_axis_y is not None:
        error = elmo_axis_y.send_command('MO=0')
        if not error:
            print_output(f"Successfully disabled Y-axis", colors='green')
        else:
            print_output(f"Failed to disable Y-axis", colors='red')


def initialize():
    global last_position_x
    global last_position_y

    last_position_x = None
    last_position_y = None

    error = ''
    if elmo_axis_x is not None:
        error += elmo_axis_x.send_command(f"MO=0")
        error += elmo_axis_x.send_command(f"SP={round(axis_parameter['SPEED_X'] / ENCODER_SCALING)}")
        error += elmo_axis_x.send_command(f"AC={round(axis_parameter['ACCEL_X'] / ENCODER_SCALING)}")
        error += elmo_axis_x.send_command(f"DC={round(axis_parameter['DECEL_X'] / ENCODER_SCALING)}")
        error += elmo_axis_x.send_command(f"SD={round(axis_parameter['STOP_DECEL_X'] / ENCODER_SCALING)}")
        error += elmo_axis_x.send_command(f"FS={FINAL_SPEED}")
        error += elmo_axis_x.send_command(f"SF={SMOOTHING_FACTOR}")
        error += elmo_axis_x.send_command(f"UM=5")
        error += elmo_axis_x.send_command(f"MO=1")
        if not error:
            print_output(f"Successfully initialized X-axis")
        else:
            print_output(f"Failed to initialize X-axis")

    error = ''
    if elmo_axis_y is not None:
        error += elmo_axis_y.send_command(f"MO=0")
        error += elmo_axis_y.send_command(f"SP={round(axis_parameter['SPEED_Y'] / ENCODER_SCALING)}")
        error += elmo_axis_y.send_command(f"AC={round(axis_parameter['ACCEL_Y'] / ENCODER_SCALING)}")
        error += elmo_axis_y.send_command(f"DC={round(axis_parameter['DECEL_Y'] / ENCODER_SCALING)}")
        error += elmo_axis_y.send_command(f"SD={round(axis_parameter['STOP_DECEL_Y'] / ENCODER_SCALING)}")
        error += elmo_axis_y.send_command(f"FS={FINAL_SPEED}")
        error += elmo_axis_y.send_command(f"SF={SMOOTHING_FACTOR}")
        error += elmo_axis_y.send_command(f"UM=5")
        error += elmo_axis_y.send_command(f"MO=1")
        if not error:
            print_output(f"Successfully initialized Y-axis")
        else:
            print_output(f"Failed to initialize Y-axis")


def commutation():
    update_progress(0)
    if elmo_axis_x is not None:
        result = calculate_commutation_offset(elmo_axis_x)
        if result != 'ERROR':
            print_output(f"Commutation of X-axis successful")
        else:
            print_output(f"Commutation of X-axis failed")
            return

    update_progress(50)
    if elmo_axis_y is not None:
        result = calculate_commutation_offset(elmo_axis_y)
        if result != 'ERROR':
            print_output(f"Commutation of Y-axis successful")
        else:
            print_output(f"Commutation of Y-axis failed")
            return


def calculate_commutation_offset(elmo_axis: ElmoAxis):
    axis = 'X-axis' if elmo_axis == elmo_axis_x else 'Y-axis'

    print_output(f"{axis} starting commutation")

    firmware_version = elmo_axis.send_command('VR', log=False)
    if firmware_version.startswith('Castanets'):
        print_output(f"{axis} running Titanium Drive commutation")
        number_of_pole_pairs_cmd = 'MP[4]'
        commutation_offset_resolution = 4096
    else:
        print_output(f"{axis} running Gold Drive commutation")
        number_of_pole_pairs_cmd = 'CA[19]'
        commutation_offset_resolution = 512

    commutation_offset = elmo_axis.send_command('CA[7]')
    print_output(f"{axis} current commutation offset: {commutation_offset}")

    # Switch motor off
    elmo_axis.send_command('MO=0')
    # Switch to stepper mode (open loop instead of closed loop encoder)
    elmo_axis.send_command('UM=3')
    # Get current torque
    torque = float(elmo_axis.send_command('CL[1]'))
    print_output(f"{axis} current torque: {torque:2.3f}")
    # Switch motor on
    elmo_axis.send_command('MO=1')
    # Torque command (increase it by 100%)
    elmo_axis.send_command(f"TC={torque * 2:2.5f}")
    # Wait 2s for axis to settle
    time.sleep(2)
    # Read position
    position = float(elmo_axis.send_command('PX'))
    print_output(f"{axis} current position: {position}")

    if elmo_axis.send_command('MO', log=False) != '1':
        print_output(f"{axis} motor is off, aborting commutation", colors='red')
        return 'ERROR'

    # Read commutation parameters
    counts_per_cycle = int(elmo_axis.send_command('CA[18]'))
    number_of_pole_pairs = int(elmo_axis.send_command(number_of_pole_pairs_cmd))
    # commutation offset = commutation angle - (position / counts per cycle * number of pole pairs * 360°)
    # in the range -360° to +360°, which are denoted as -512 to +512 (Gold) or -4096 to +4096 (Platinum and Titanium)
    # commutation angle = stepper angle + 90° (with stepper angle = 0° in this case)
    commutation_offset_angle = (0 + 90) - (position / counts_per_cycle * number_of_pole_pairs * 360)
    commutation_offset = round(commutation_offset_angle / 360 * commutation_offset_resolution) % commutation_offset_resolution
    print_output(f"{axis} new commutation offset: {commutation_offset}")
    # Switch motor off
    elmo_axis.send_command('MO=0')
    time.sleep(1)
    # Save the new commutation offset
    elmo_axis.send_command(f"CA[7]={commutation_offset}")
    elmo_axis.send_command('SV')
    time.sleep(1)

    return commutation_offset


def download_parameters_file():
    file_path = sg.popup_get_file('Select Drive Parameters File', no_window=True, file_types=(('PPRM File', '*.pprm'), ('GPRM File', '*.gprm')))
    if not file_path:
        return

    firmware_version = ''
    axis_encoding = 0
    file_header: list[str] = []
    parameter_list: list[str] = []

    with open(file_path, 'r') as file:
        for line in file:
            line = line.strip()
            if line.startswith('% Drive FW Version:'):
                firmware_version = line.split(':', maxsplit=1)[1].strip()
            elif line.startswith('UI[2]'):
                axis_encoding = int(line.split('=', maxsplit=1)[1].strip())
            if line.startswith('%'):
                file_header.append(line)
            elif line != '':
                parameter_list.append(line)

    print_output(f"Drive Parameters File Header:")
    for line in file_header:
        print_output(line)

    if not firmware_version:
        print_output(f"Failed to read firmware version from file: {file_path}", colors='red')
        return

    print_output(f"Firmware version from file: {firmware_version}")
    print_output(f"Axis encoding from file: {axis_encoding}")

    elmo_drive = None
    firmware_name = ''
    if firmware_version.startswith('Castanet'):
        firmware_name = 'Castanet'
        if elmo_axis_x is not None:
            elmo_drive = elmo_axis_x.drive
    if firmware_version.startswith('Whistle'):
        firmware_name = 'Whistle'
        if axis_encoding == AXIS_ENCODING_X and elmo_axis_x is not None:
            elmo_drive = elmo_axis_x.drive
        if axis_encoding == AXIS_ENCODING_Y and elmo_axis_y is not None:
            elmo_drive = elmo_axis_y.drive

    if elmo_drive is None:
        print_output(f"No matching Elmo drive found", colors='red')
        return

    drive_firmware_version = elmo_drive.send_command('VR', log=False)
    if not drive_firmware_version.startswith(firmware_name):
        print_output(f"Firmware version mismatch: File '{firmware_version}' vs Drive '{drive_firmware_version}'", colors='red')
        return

    choice = sg.popup_yes_no(
        f"Download parameters to drive '{elmo_drive.name}'?",
        f"Drive Firmware Version:",
        f"\t{drive_firmware_version}",
        title='Download Parameters', keep_on_top=True
    )
    if choice != 'Yes':
        print_output(f"Download cancelled by user", colors='orange')
        return

    stop()

    # No factory reset so that the ethernet communication settings are preserved
    # A production programming tool using USB direct access could use the following command to reset to factory defaults:
    # elmo_drive.send_command('RS')

    print_output(f"Downloading parameters to drive...")
    failed_parameters = []
    for parameter in parameter_list:
        if parameter.startswith('CA[7]'):
            # Skip commutation offset, as it must be set by the commutation
            continue
        error = elmo_drive.send_command(parameter)
        if error:
            failed_parameters.append(parameter)

    if failed_parameters:
        print_output(f"Failed to set {len(failed_parameters)} parameters:", colors='red')
        output = ''
        for i, parameter in enumerate(failed_parameters):
            if i >= 10:
                output += f", ... ({len(failed_parameters) - i} more)"
                break
            output += parameter if i == 0 else f", {parameter}"
        print_output(output, colors='red')
        return

    elmo_drive.send_command('SV')  # Save parameters to drive
    print_output(f"Parameters downloaded successfully to drive", colors='green')


def move_x(position, current_scan_enable=False):
    global last_position_x

    if elmo_axis_x is None:
        raise ElmoAxisError("X-axis not found")

    print_output(f"X-axis moving to {position:.6f} mm")

    elmo_axis_x.send_command(f"PA={position / ENCODER_SCALING:.0f}")
    elmo_axis_x.send_command(f"BG")

    start_moving_time = time.perf_counter()
    start_settling_time = None
    start_idle_time = None
    motion_status = None
    current_scan = {}
    current_scan_mod = 0

    while motion_status != MOTOR_IDLE:
        motion_status = int(elmo_axis_x.send_command('MS', log=False))
        current_time = time.perf_counter()

        if motion_status == MOTOR_SETTLING and start_settling_time is None:
            start_settling_time = current_time

        if motion_status == MOTOR_DISABLED:
            raise ElmoAxisError("X-axis motor is disabled")
        if current_time - start_moving_time > MOTION_TIMEOUT:
            if STOP_ON_TIMEOUT:
                stop()
            raise ElmoAxisError("Motion Timeout")

        time.sleep(0.01)  # 10 ms delay to avoid excessive polling

        # Record current scan data every 10th poll when enabled
        if current_scan_enable and motion_status == MOTOR_MOVING and (current_scan_mod % 10 == 0):
            current_pos = float(elmo_axis_x.send_command('PU', log=False)) * ENCODER_SCALING
            current = float(elmo_axis_x.send_command('IQ', log=False))
            current_scan[current_pos] = current

        current_scan_mod += 1

    update_position()  # Update current position

    start_idle_time = current_time
    if start_settling_time is None:
        start_settling_time = start_idle_time

    settling_time = start_idle_time - start_settling_time
    moving_time = start_settling_time - start_moving_time
    total_time = start_idle_time - start_moving_time

    print_output(f"X-axis target reached in {total_time * 1000:.0f} ms")
    print_output(f"X-axis settled in {settling_time * 1000:.0f} ms")
    print_output('')

    last_position_x = position
    results = {
        'total_time_x': total_time, 'moving_time_x': moving_time, 'settling_time_x': settling_time, 'current_scan_x': current_scan,
    }
    return results


def move_y(position, current_scan_enable=False):
    global last_position_y

    if elmo_axis_y is None:
        raise ElmoAxisError("Y-axis not found")

    print_output(f"Y-axis moving to {position:.6f} mm")

    elmo_axis_y.send_command(f"PA={position / ENCODER_SCALING:.0f}")
    elmo_axis_y.send_command(f"BG")

    start_moving_time = time.perf_counter()
    start_settling_time = None
    start_idle_time = None
    motion_status = None
    current_scan = {}
    current_scan_mod = 0

    while motion_status != MOTOR_IDLE:
        motion_status = int(elmo_axis_y.send_command('MS', log=False))
        current_time = time.perf_counter()

        if motion_status == MOTOR_SETTLING and start_settling_time is None:
            start_settling_time = current_time

        if motion_status == MOTOR_DISABLED:
            raise ElmoAxisError("Y-axis motor is disabled")
        if (current_time - start_moving_time) > MOTION_TIMEOUT:
            if STOP_ON_TIMEOUT:
                stop()
            raise ElmoAxisError("Motion Timeout")

        time.sleep(0.01)  # 10 ms delay to avoid excessive polling

        # Record current scan data every 10th poll when enabled
        if current_scan_enable and motion_status == MOTOR_MOVING and (current_scan_mod % 10 == 0):
            current_pos = float(elmo_axis_y.send_command('PU', log=False)) * ENCODER_SCALING
            current = float(elmo_axis_y.send_command('IQ', log=False))
            current_scan[current_pos] = current

        current_scan_mod += 1

    update_position()  # Update current position

    start_idle_time = current_time
    if start_settling_time is None:
        start_settling_time = start_idle_time

    settling_time = start_idle_time - start_settling_time
    moving_time = start_settling_time - start_moving_time
    total_time = start_idle_time - start_moving_time

    print_output(f"Y-axis target reached in {total_time * 1000:.0f} ms")
    print_output(f"Y-axis settled in {settling_time * 1000:.0f} ms")
    print_output('')

    last_position_y = position
    results = {
        'total_time_y': total_time, 'moving_time_y': moving_time, 'settling_time_y': settling_time, 'current_scan_y': current_scan,
    }
    return results


def move(position_x, position_y):
    global last_position_x
    global last_position_y

    if elmo_axis_x is None or elmo_axis_y is None:
        raise ElmoAxisError("X-axis and/or Y-axis not found")
    if position_x == last_position_x and position_y == last_position_y:
        print_output(f"Already at target position")
        return {}
    if position_x != last_position_x and position_y == last_position_y:
        return move_x(position_x)
    if position_x == last_position_x and position_y != last_position_y:
        return move_y(position_y)

    print_output(f"X-axis moving to {position_x:.6f} mm")
    print_output(f"Y-axis moving to {position_y:.6f} mm")

    elmo_axis_x.send_command(f"PA={position_x / ENCODER_SCALING:.0f}")
    elmo_axis_y.send_command(f"PA={position_y / ENCODER_SCALING:.0f}")
    elmo_axis_x.send_command(f"BG")
    elmo_axis_y.send_command(f"BG")

    start_moving_time = time.perf_counter()
    start_settling_time_x = None
    start_settling_time_y = None
    start_idle_time_x = None
    start_idle_time_y = None
    motion_status_x = None
    motion_status_y = None

    while motion_status_x != MOTOR_IDLE or motion_status_y != MOTOR_IDLE:
        motion_status_x = int(elmo_axis_x.send_command('MS', log=False))
        motion_status_y = int(elmo_axis_y.send_command('MS', log=False))
        current_time = time.perf_counter()

        if motion_status_x == MOTOR_SETTLING and start_settling_time_x is None:
            start_settling_time_x = current_time
        if motion_status_y == MOTOR_SETTLING and start_settling_time_y is None:
            start_settling_time_y = current_time

        if motion_status_x == MOTOR_IDLE and start_idle_time_x is None:
            start_idle_time_x = current_time
        if motion_status_y == MOTOR_IDLE and start_idle_time_y is None:
            start_idle_time_y = current_time

        if motion_status_x == MOTOR_DISABLED or motion_status_y == MOTOR_DISABLED:
            raise ElmoAxisError("X-axis and/or Y-axis motor is disabled")
        if current_time - start_moving_time > MOTION_TIMEOUT:
            if STOP_ON_TIMEOUT:
                stop()
            raise ElmoAxisError("Motion Timeout")

        time.sleep(0.01)  # 10 ms delay to avoid excessive polling

    update_position()  # Update current position

    if start_settling_time_x is None:
        start_settling_time_x = start_idle_time_x
    if start_settling_time_y is None:
        start_settling_time_y = start_idle_time_y

    settling_time_x = start_idle_time_x - start_settling_time_x
    settling_time_y = start_idle_time_y - start_settling_time_y
    moving_time_x = start_settling_time_x - start_moving_time
    moving_time_y = start_settling_time_y - start_moving_time
    total_time_x = start_idle_time_x - start_moving_time
    total_time_y = start_idle_time_y - start_moving_time

    print_output(f"X-axis target reached in {total_time_x * 1000:.0f} ms")
    print_output(f"X-axis settled in {settling_time_x * 1000:.0f} ms")
    print_output(f"Y-axis target reached in {total_time_y * 1000:.0f} ms")
    print_output(f"Y-axis settled in {settling_time_y * 1000:.0f} ms")
    print_output('')

    last_position_x = position_x
    last_position_y = position_y
    results = {
        'total_time_x': total_time_x, 'moving_time_x': moving_time_x, 'settling_time_x': settling_time_x,
        'total_time_y': total_time_y, 'moving_time_y': moving_time_y, 'settling_time_y': settling_time_y,
    }
    return results


def run_plate_scan(plate_scan_type, iterations, file_prefix):
    print_output(f"Starting Plate Scan with {plate_scan_type}")

    if elmo_axis_x is None or elmo_axis_y is None:
        print_output(f"X-axis and/or Y-axis not found", colors='red')
        return

    if axis_parameter is None:
        print_output(f"Scan table is not initialized", colors='red')
        return

    update_progress(0)

    plate_scan_datetime = datetime.now()
    plate_scan_results = {
        'date': plate_scan_datetime.strftime('%Y-%m-%d %H:%M:%S'),
        'scan_type': plate_scan_type,
        'speed_x': axis_parameter['SPEED_X'],
        'accel_x': axis_parameter['ACCEL_X'],
        'decel_x': axis_parameter['DECEL_X'],
        'speed_y': axis_parameter['SPEED_Y'],
        'accel_y': axis_parameter['ACCEL_Y'],
        'decel_y': axis_parameter['DECEL_Y'],
        'iterations': iterations,
        'total_time': 0,
        'settling_time_x': {},
        'settling_time_y': {},
    }

    total_well_count = PLATE_ROWS * PLATE_COLUMNS * iterations
    well_count = 0

    settling_time_x: dict = plate_scan_results['settling_time_x']
    settling_time_y: dict = plate_scan_results['settling_time_y']

    start_position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X
    if plate_scan_type in ['Well Meander Y', 'Well Meander Only X']:
        start_position_x -= PLATE_WELL_X
    start_position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y
    if plate_scan_type in ['Well Meander X', 'Well Meander Only Y']:
        start_position_y -= PLATE_WELL_Y

    try:
        # Move to start position
        move(start_position_x, start_position_y)

        total_scan_start_time = time.perf_counter()

        for i in range(iterations):
            iteration_start_time = time.perf_counter()
            print_output(f"Iteration {i + 1} / {iterations}")

            if plate_scan_type == 'Well Meander X':
                for row in range(PLATE_ROWS):
                    col_range = range(PLATE_COLUMNS)
                    if row % 2 == 1:
                        col_range = reversed(col_range)
                    for col in col_range:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + (col * PLATE_WELL_X)
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + (row * PLATE_WELL_Y)
                        result = move(position_x, position_y)
                        if 'settling_time_x' in result:
                            settling_time_x[(i, col, row)] = result['settling_time_x']
                        if 'settling_time_y' in result:
                            settling_time_y[(i, col, row)] = result['settling_time_y']
                        well_count += 1
                        update_progress(well_count / total_well_count * 100)

            elif plate_scan_type == 'Well Meander Y':
                for col in range(PLATE_COLUMNS):
                    row_range = range(PLATE_ROWS)
                    if col % 2 == 1:
                        row_range = reversed(row_range)
                    for row in row_range:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + (col * PLATE_WELL_X)
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + (row * PLATE_WELL_Y)
                        result = move(position_x, position_y)
                        if 'settling_time_x' in result:
                            settling_time_x[(i, col, row)] = result['settling_time_x']
                        if 'settling_time_y' in result:
                            settling_time_y[(i, col, row)] = result['settling_time_y']
                        well_count += 1
                        update_progress(well_count / total_well_count * 100)

            if plate_scan_type == 'Well Meander Only X':
                for row in range(PLATE_ROWS):
                    col_range = range(PLATE_COLUMNS)
                    if row % 2 == 1:
                        col_range = reversed(col_range)
                    for col in col_range:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + (col * PLATE_WELL_X)
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + (row * PLATE_WELL_Y)
                        result = move(position_x, position_y)
                        if 'settling_time_x' in result:
                            settling_time_x[(i, col, row)] = result['settling_time_x']
                        if 'settling_time_y' in result:
                            settling_time_y[(i, col, row)] = result['settling_time_y']
                        well_count += 1
                        update_progress(well_count / total_well_count * 100)

                    position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + ((row + 1) * PLATE_WELL_Y)
                    if col > 0:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + ((col + 1) * PLATE_WELL_X)
                    else:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + ((col - 1) * PLATE_WELL_X)

                    move(position_x, position_y)

            elif plate_scan_type == 'Well Meander Only Y':
                for col in range(PLATE_COLUMNS):
                    row_range = range(PLATE_ROWS)
                    if col % 2 == 1:
                        row_range = reversed(row_range)
                    for row in row_range:
                        position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + (col * PLATE_WELL_X)
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + (row * PLATE_WELL_Y)
                        result = move(position_x, position_y)
                        if 'settling_time_x' in result:
                            settling_time_x[(i, col, row)] = result['settling_time_x']
                        if 'settling_time_y' in result:
                            settling_time_y[(i, col, row)] = result['settling_time_y']
                        well_count += 1
                        update_progress(well_count / total_well_count * 100)

                    position_x = PLATE_ORIGIN_X + ORIGIN_OFFSET_X + ((col + 1) * PLATE_WELL_X)
                    if row > 0:
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + ((row + 1) * PLATE_WELL_Y)
                    else:
                        position_y = PLATE_ORIGIN_Y + ORIGIN_OFFSET_Y + ((row - 1) * PLATE_WELL_Y)

                    move(position_x, position_y)

            # Return to start position
            move(start_position_x, start_position_y)

            interation_end_time = time.perf_counter()
            print_output(f"Iteration {i + 1} completed in {interation_end_time - iteration_start_time:.3f} s")

    except ElmoAxisError as e:
        print_output(f"Plate Scan aborted: {e}", colors='red')
        return

    total_scan_end_time = time.perf_counter()
    plate_scan_results['total_time'] = total_scan_end_time - total_scan_start_time

    file_name = f"plate_scan_{plate_scan_datetime.strftime('%Y%m%d_%H%M%S')}"
    if file_prefix != '':
        file_name = f"{file_prefix}_{file_name}"

    save_plate_scan_results(plate_scan_results, file_name)
    main_window.write_event_value('-PSCANRESULT-', (plate_scan_results, file_name))


def save_plate_scan_results(results, file_name):
    with open(os.path.join(PLATE_SCAN_RESULTS_DIR, f"{file_name}.json"), 'w') as file:
        # Convert keys of settling_time_x and settling_time_y to strings for JSON serialization
        results_copy = dict(results)
        results_copy['settling_time_x'] = {str(key): value for key, value in results['settling_time_x'].items()}
        results_copy['settling_time_y'] = {str(key): value for key, value in results['settling_time_y'].items()}
        json.dump(results_copy, file, indent=4)

    print_output(f"Plate Scan Results saved as {file_name}.json", colors='green')

    plate_scan_type = results['scan_type']
    plate_scan_iterations = results['iterations']
    plate_scan_total_time = results['total_time']
    settling_time_x: dict = results['settling_time_x']
    settling_time_y: dict = results['settling_time_y']

    with open(PLATE_SCAN_RESULTS_LOG, 'a') as file:
        file.write(f"Plate Scan {results['date']}\n")
        file.write(f"Scan Type: {plate_scan_type}\n")
        file.write(f"Speed X: {results['speed_x']:.1f}\n")
        file.write(f"Accel X: {results['accel_x']:.1f}\n")
        file.write(f"Decel X: {results['decel_x']:.1f}\n")
        file.write(f"Speed Y: {results['speed_y']:.1f}\n")
        file.write(f"Accel Y: {results['accel_y']:.1f}\n")
        file.write(f"Decel Y: {results['decel_y']:.1f}\n")
        file.write(f"Nr. of Iterations: {plate_scan_iterations}\n")
        file.write('\n')

        # for i in range(plate_scan_iterations):
        #     file.write(f"Iteration {i + 1} / {plate_scan_iterations}:\n")
        #     file.write(f"Settling Time [ms]:\n")
        #     file.write(f"; Well ; X    ; Y    ;\n")
        #     if plate_scan_type in ['Well Meander X', 'Well Meander Only X']:
        #         for row in range(PLATE_ROWS):
        #             col_range = range(PLATE_COLUMNS)
        #             if row % 2 == 1:
        #                 col_range = reversed(col_range)
        #             for col in col_range:
        #                 file.write(f"; {chr(65 + row)}{col + 1:<3d} ;")
        #                 if (i, col, row) in settling_time_x:
        #                     file.write(f" {settling_time_x[(i, col, row)] * 1000:4.0f} ;")
        #                 else:
        #                     file.write(f" {0.0:4.0f} ;")
        #                 if (i, col, row) in settling_time_y:
        #                     file.write(f" {settling_time_y[(i, col, row)] * 1000:4.0f} ;")
        #                 else:
        #                     file.write(f" {0.0:4.0f} ;")
        #                 file.write('\n')
        #     elif plate_scan_type in ['Well Meander Y', 'Well Meander Only Y']:
        #         for col in range(PLATE_COLUMNS):
        #             row_range = range(PLATE_ROWS)
        #             if col % 2 == 1:
        #                 row_range = reversed(row_range)
        #             for row in row_range:
        #                 file.write(f"; {chr(65 + row)}{col + 1:<3d} ;")
        #                 if (i, col, row) in settling_time_x:
        #                     file.write(f" {settling_time_x[(i, col, row)] * 1000:4.0f} ;")
        #                 else:
        #                     file.write(f" {0.0:4.0f} ;")
        #                 if (i, col, row) in settling_time_y:
        #                     file.write(f" {settling_time_y[(i, col, row)] * 1000:4.0f} ;")
        #                 else:
        #                     file.write(f" {0.0:4.0f} ;")
        #                 file.write('\n')

        #     file.write('\n')

        file.write(f"Mean Settling Time X-axis [ms]:\n")
        file.write(f";   ;")
        for col in range(PLATE_COLUMNS):
            file.write(f" {col + 1:<4d} ;")
        file.write('\n')
        for row in range(PLATE_ROWS):
            file.write(f"; {chr(65 + row)} ;")
            for col in range(PLATE_COLUMNS):
                settling_time_x_mean = 0
                for i in range(plate_scan_iterations):
                    settling_time_x_mean += settling_time_x.get((i, col, row), 0)
                settling_time_x_mean /= plate_scan_iterations
                file.write(f" {settling_time_x_mean * 1000:4.0f} ;")
            file.write('\n')
        file.write('\n')

        file.write(f"Max Settling Time X-axis [ms]:\n")
        file.write(f";   ;")
        for col in range(PLATE_COLUMNS):
            file.write(f" {col + 1:<4d} ;")
        file.write('\n')
        for row in range(PLATE_ROWS):
            file.write(f"; {chr(65 + row)} ;")
            for col in range(PLATE_COLUMNS):
                settling_time_x_max = 0
                for i in range(plate_scan_iterations):
                    settling_time_x_max = max(settling_time_x_max, settling_time_x.get((i, col, row), 0))
                file.write(f" {settling_time_x_max * 1000:4.0f} ;")
            file.write('\n')
        file.write('\n')

        file.write(f"Mean Settling Time Y-axis [ms]:\n")
        file.write(f";   ;")
        for col in range(PLATE_COLUMNS):
            file.write(f" {col + 1:<4d} ;")
        file.write('\n')
        for row in range(PLATE_ROWS):
            file.write(f"; {chr(65 + row)} ;")
            for col in range(PLATE_COLUMNS):
                settling_time_y_mean = 0
                for i in range(plate_scan_iterations):
                    settling_time_y_mean += settling_time_y.get((i, col, row), 0)
                settling_time_y_mean /= plate_scan_iterations
                file.write(f" {settling_time_y_mean * 1000:4.0f} ;")
            file.write('\n')
        file.write('\n')

        file.write(f"Max Settling Time Y-axis [ms]:\n")
        file.write(f";   ;")
        for col in range(PLATE_COLUMNS):
            file.write(f" {col + 1:<4d} ;")
        file.write('\n')
        for row in range(PLATE_ROWS):
            file.write(f"; {chr(65 + row)} ;")
            for col in range(PLATE_COLUMNS):
                settling_time_y_max = 0
                for i in range(plate_scan_iterations):
                    settling_time_y_max = max(settling_time_y_max, settling_time_y.get((i, col, row), 0))
                file.write(f" {settling_time_y_max * 1000:4.0f} ;")
            file.write('\n')
        file.write('\n')

        file.write(f"Total plate scan time: {plate_scan_total_time:.3f} s\n")
        file.write(f"Mean time per iteration: {plate_scan_total_time / plate_scan_iterations:.3f} s\n")

        if settling_time_x:
            settling_time_x_count = len(settling_time_x)
            settling_time_x_mean = sum(settling_time_x.values()) / len(settling_time_x)
            settling_time_x_min = min(settling_time_x.values())
            settling_time_x_max = max(settling_time_x.values())
            file.write(
                f"X-axis settling time statistics [ms]:"
                f" count = {settling_time_x_count:5d},"
                f" mean = {settling_time_x_mean * 1000:4.0f},"
                f" min = {settling_time_x_min * 1000:4.0f},"
                f" max = {settling_time_x_max * 1000:4.0f}\n"
            )
        else:
            file.write(f"X-axis settling time statistics [ms]: count =     0, mean =    0, min =    0, max =    0\n")

        if settling_time_y:
            settling_time_y_count = len(settling_time_y)
            settling_time_y_mean = sum(settling_time_y.values()) / len(settling_time_y)
            settling_time_y_min = min(settling_time_y.values())
            settling_time_y_max = max(settling_time_y.values())
            file.write(
                f"Y-axis settling time statistics [ms]:"
                f" count = {settling_time_y_count:5d},"
                f" mean = {settling_time_y_mean * 1000:4.0f},"
                f" min = {settling_time_y_min * 1000:4.0f},"
                f" max = {settling_time_y_max * 1000:4.0f}\n"
            )
        else:
            file.write(f"Y-axis settling time statistics [ms]: count =     0, mean =    0, min =    0, max =    0\n")

        file.write('\n\n')


def load_plate_scan_results():
    file_path = sg.popup_get_file('Select Plate Scan Results File', no_window=True, file_types=(('JSON Files', '*.json'),))
    if not file_path:
        return

    try:
        with open(file_path, 'r') as file:
            plate_scan_results = json.load(file)

        plate_scan_results['settling_time_x'] = {literal_eval(key): value for key, value in plate_scan_results['settling_time_x'].items()}
        plate_scan_results['settling_time_y'] = {literal_eval(key): value for key, value in plate_scan_results['settling_time_y'].items()}
    except Exception as e:
        print_output(f"Failed to load Plate Scan Results: {e}", colors='red')
        return

    print_output(f"Plate Scan Results loaded from {file_path}", colors='green')
    show_plate_scan_results_map(plate_scan_results)
    print_plate_scan_results_summary(plate_scan_results)


def print_plate_scan_results_summary(results):
    plate_scan_iterations = results['iterations']
    plate_scan_total_time = results['total_time']
    settling_time_x: dict = results['settling_time_x']
    settling_time_y: dict = results['settling_time_y']

    print_output(f"Plate Scan completed in {plate_scan_total_time:.3f} s")
    print_output(f"Mean time per iteration: {plate_scan_total_time / plate_scan_iterations:.3f} s")

    if settling_time_x:
        settling_time_x_count = len(settling_time_x)
        settling_time_x_mean = sum(settling_time_x.values()) / len(settling_time_x)
        settling_time_x_min = min(settling_time_x.values())
        settling_time_x_max = max(settling_time_x.values())
        print_output(f"X-axis movement count: {settling_time_x_count}")
        print_output(f"X-axis settling time [ms]:")
        print_output(f"    mean = {settling_time_x_mean * 1000:4.0f}, min = {settling_time_x_min * 1000:4.0f}, max = {settling_time_x_max * 1000:4.0f}")
    else:
        print_output(f"X-axis movement count: 0")
        print_output(f"X-axis settling time [ms]:")
        print_output(f"    mean =    0, min =    0, max =    0")

    if settling_time_y:
        settling_time_y_count = len(settling_time_y)
        settling_time_y_mean = sum(settling_time_y.values()) / len(settling_time_y)
        settling_time_y_min = min(settling_time_y.values())
        settling_time_y_max = max(settling_time_y.values())
        print_output(f"Y-axis movement count: {settling_time_y_count}")
        print_output(f"Y-axis settling time [ms]:")
        print_output(f"    mean = {settling_time_y_mean * 1000:4.0f}, min = {settling_time_y_min * 1000:4.0f}, max = {settling_time_y_max * 1000:4.0f}")
    else:
        print_output(f"Y-axis movement count: 0")
        print_output(f"Y-axis settling time [ms]:")
        print_output(f"    mean =    0, min =    0, max =    0")


def show_plate_scan_results_map(results, file_name=None):
    plate_scan_datetime = results['date']
    plate_scan_type = results['scan_type']
    plate_scan_iterations = results['iterations']
    settling_time_x: dict = results['settling_time_x']
    settling_time_y: dict = results['settling_time_y']

    settling_time_x_mean = np.zeros((PLATE_ROWS, PLATE_COLUMNS))
    settling_time_y_mean = np.zeros((PLATE_ROWS, PLATE_COLUMNS))
    settling_time_x_max = np.zeros((PLATE_ROWS, PLATE_COLUMNS))
    settling_time_y_max = np.zeros((PLATE_ROWS, PLATE_COLUMNS))

    for i in range(plate_scan_iterations):
        for row in range(PLATE_ROWS):
            for col in range(PLATE_COLUMNS):
                settling_time_x_value = settling_time_x.get((i, col, row), 0)
                settling_time_y_value = settling_time_y.get((i, col, row), 0)
                settling_time_x_mean[row, col] += settling_time_x_value
                settling_time_y_mean[row, col] += settling_time_y_value
                settling_time_x_max[row, col] = max(settling_time_x_max[row, col], settling_time_x_value)
                settling_time_y_max[row, col] = max(settling_time_y_max[row, col], settling_time_y_value)

    settling_time_x_mean /= plate_scan_iterations
    settling_time_y_mean /= plate_scan_iterations

    settling_time_x_mean[settling_time_x_mean == 0] = float('nan')
    settling_time_y_mean[settling_time_y_mean == 0] = float('nan')
    settling_time_x_max[settling_time_x_max == 0] = float('nan')
    settling_time_y_max[settling_time_y_max == 0] = float('nan')

    figure_title = f"Plate Scan Results Map {plate_scan_datetime}"
    plt.close(figure_title)
    plt.figure(figure_title)
    plt.suptitle(f"{plate_scan_type} - {plate_scan_datetime}\nIterations: {plate_scan_iterations}", fontsize=15)

    plt.subplot(2, 2, 1)
    plt.imshow(settling_time_x_mean * 1000, cmap='RdYlGn_r', vmin=PLATE_SCAN_CMAP_MIN, vmax=PLATE_SCAN_CMAP_MAX, interpolation='nearest')
    plt.title(f"Mean Settling Time X-axis [ms]")
    plt.xlabel('Columns')
    plt.ylabel('Rows')

    plt.subplot(2, 2, 2)
    plt.imshow(settling_time_y_mean * 1000, cmap='RdYlGn_r', vmin=PLATE_SCAN_CMAP_MIN, vmax=PLATE_SCAN_CMAP_MAX, interpolation='nearest')
    plt.title(f"Mean Settling Time Y-axis [ms]")
    plt.xlabel('Columns')
    plt.ylabel('Rows')

    plt.subplot(2, 2, 3)
    plt.imshow(settling_time_x_max * 1000, cmap='RdYlGn_r', vmin=PLATE_SCAN_CMAP_MIN, vmax=PLATE_SCAN_CMAP_MAX, interpolation='nearest')
    plt.title(f"Max Settling Time X-axis [ms]")
    plt.xlabel('Columns')
    plt.ylabel('Rows')

    plt.subplot(2, 2, 4)
    plt.imshow(settling_time_y_max * 1000, cmap='RdYlGn_r', vmin=PLATE_SCAN_CMAP_MIN, vmax=PLATE_SCAN_CMAP_MAX, interpolation='nearest')
    plt.title(f"Max Settling Time Y-axis [ms]")
    plt.xlabel('Columns')
    plt.ylabel('Rows')

    plt.tight_layout(h_pad=2.0)

    if file_name is not None:
        plt.savefig(os.path.join(PLATE_SCAN_RESULTS_DIR, f"{file_name}.png"))
        print_output(f"Plate Scan Results Map saved as {file_name}.png", colors='green')

    plt.show(block=False)


def show_plate_scan_results_log():
    if os.path.exists(PLATE_SCAN_RESULTS_LOG):
        sg.popup_scrolled(open(PLATE_SCAN_RESULTS_LOG).read(), title='Plate Scan Results Log', size=(175, 40), font='Courier 10')
    else:
        print_output(f"No Plate Scan Results Log found", colors='red')


def run_production_test(iterations):
    global FIXED_POS_X
    global FIXED_POS_Y
    global settling_time_x
    global settling_time_y
    global move_count_x
    global move_count_y

    FIXED_POS_X =  2.0
    FIXED_POS_Y = 77.0

    START_POS_X   =   2.0
    END_POS_X     = 375.0
    PLATE_DOOR_X  = 200.0
    START_POS_Y   =  -3.0
    END_POS_Y     =  80.0
    MIN_STEP_SIZE =  14.0

    print_output(f"Starting Production Test")

    if elmo_axis_x is None or elmo_axis_y is None:
        print_output(f"X-axis and/or Y-axis not found", colors='red')
        return

    if axis_parameter is None:
        print_output(f"Scan table is not initialized", colors='red')
        return

    update_progress(0)

    settling_time_x = {}
    settling_time_y = {}
    move_count_x = 0
    move_count_y = 0

    try:
        table_edge_test()

        total_start_time = time.perf_counter()

        for i in range(iterations):
            start_time = time.perf_counter()
            print_output(f"Iteration {i + 1} / {iterations}")

            table_single_axis_test('x', START_POS_X, END_POS_X, MIN_STEP_SIZE)
            table_single_axis_test('y', START_POS_Y, END_POS_Y, MIN_STEP_SIZE)
            table_diag_move_test(START_POS_X, PLATE_DOOR_X, START_POS_Y, END_POS_Y, MIN_STEP_SIZE)

            end_time = time.perf_counter()
            print_output(f"Iteration {i + 1} completed in {end_time - start_time:.3f} s")
            update_progress((i + 1) / iterations * 100)


        total_end_time = time.perf_counter()
        print_output(f"Production Test completed in {total_end_time - total_start_time:.3f} s")
        print_output(f"Mean time per iteration: {(total_end_time - total_start_time) / iterations:.3f} s")

        if settling_time_x:
            settling_time_x_count = len(settling_time_x)
            settling_time_x_mean = sum(settling_time_x.values()) / len(settling_time_x)
            settling_time_x_min = min(settling_time_x.values())
            settling_time_x_max = max(settling_time_x.values())
            print_output(f"X-axis movement count: {settling_time_x_count}")
            print_output(f"X-axis settling time [ms]:")
            print_output(f"    mean = {settling_time_x_mean * 1000:4.0f}, min = {settling_time_x_min * 1000:4.0f}, max = {settling_time_x_max * 1000:4.0f}")
        else:
            print_output(f"X-axis movement count: 0")
            print_output(f"X-axis settling time [ms]:")
            print_output(f"    mean =    0, min =    0, max =    0")

        if settling_time_y:
            settling_time_y_count = len(settling_time_y)
            settling_time_y_mean = sum(settling_time_y.values()) / len(settling_time_y)
            settling_time_y_min = min(settling_time_y.values())
            settling_time_y_max = max(settling_time_y.values())
            print_output(f"Y-axis movement count: {settling_time_y_count}")
            print_output(f"Y-axis settling time [ms]:")
            print_output(f"    mean = {settling_time_y_mean * 1000:4.0f}, min = {settling_time_y_min * 1000:4.0f}, max = {settling_time_y_max * 1000:4.0f}")
        else:
            print_output(f"Y-axis movement count: 0")
            print_output(f"Y-axis settling time [ms]:")
            print_output(f"    mean =    0, min =    0, max =    0")

    except ElmoAxisError as e:
        print_output(f"Production Test aborted: {e}", colors='red')
        return


def table_edge_test():
    global FIXED_POS_X
    global FIXED_POS_Y
    global settling_time_x
    global settling_time_y
    global move_count_x
    global move_count_y

    MIN_POS_X =   8.0
    MAX_POS_X = 118.5
    MIN_POS_Y =   0.7
    MAX_POS_Y =  79.0
    END_POS_X = 374.9

    print_output(f"Starting Edge Test")

    # Top Left
    production_test_move(MIN_POS_X, MIN_POS_Y)
    # Top Right
    production_test_move(MAX_POS_X, MIN_POS_Y)
    # Plate Door
    production_test_move(MAX_POS_X, FIXED_POS_Y)
    # Transfer
    production_test_move(END_POS_X, FIXED_POS_Y)
    # Plate Door
    production_test_move(MAX_POS_X, FIXED_POS_Y)
    # Bottom Right
    production_test_move(MAX_POS_X, MAX_POS_Y)
    # Bottom Left
    production_test_move(MIN_POS_X, MAX_POS_Y)

    print_output(f"Edge Test finished successfully", colors='green')


def table_single_axis_test(axis: str, start_pos, end_pos, min_step_size):
    global FIXED_POS_X
    global FIXED_POS_Y

    total_dist = end_pos - start_pos
    step_count = int(total_dist / min_step_size)
    step_size = total_dist / step_count

    # Move to start pos
    if axis.lower() == 'x':
        production_test_move(start_pos, FIXED_POS_Y)
    else:
        production_test_move(FIXED_POS_X, start_pos)

    new_pos = start_pos

    for i in range(step_count):
        new_pos += 0.1
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

        new_pos += 0.9
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

        new_pos += step_size - 1.0
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

    for i in range(step_count):
        new_pos -= 0.1
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

        new_pos -= 0.9
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

        new_pos -= step_size - 1.0
        if axis.lower() == 'x':
            production_test_move(new_pos, FIXED_POS_Y)
        else:
            production_test_move(FIXED_POS_X, new_pos)

    if axis.lower() == 'x':
        production_test_move(start_pos, FIXED_POS_Y)
    else:
        production_test_move(FIXED_POS_X, start_pos)

    if axis.lower() == 'x':
        production_test_move(end_pos, FIXED_POS_Y)
    else:
        production_test_move(FIXED_POS_X, end_pos)

    if axis.lower() == 'x':
        production_test_move(start_pos, FIXED_POS_Y)
    else:
        production_test_move(FIXED_POS_X, start_pos)


def table_diag_move_test(start_pos_x, end_pos_x, start_pos_y, end_pos_y, min_step_size):
    # This Test assumes that the x distance is much bigger (at least double) than the y distance
    total_dist_x = end_pos_x - start_pos_x
    total_dist_y = end_pos_y - start_pos_y
    step_count_x = int(total_dist_x / min_step_size)
    step_count_y = int(total_dist_y / min_step_size)
    step_count = min(step_count_x, step_count_y)
    step_size_x = total_dist_x / step_count / 2
    step_size_y = total_dist_y / step_count

    # Move to start pos
    production_test_move(start_pos_x, start_pos_y)

    new_pos_x = start_pos_x
    new_pos_y = start_pos_y
    dir_x = 1
    dir_y = 1

    for j in range(4):
        for i in range(step_count):
            new_pos_x += 0.1 * dir_x
            new_pos_y += 0.1 * dir_y
            production_test_move(new_pos_x, new_pos_y)

            new_pos_x += 0.9 * dir_x
            new_pos_y += 0.9 * dir_y
            production_test_move(new_pos_x, new_pos_y)

            new_pos_x += (step_size_x - 1.0) * dir_x
            new_pos_y += (step_size_y - 1.0) * dir_y
            production_test_move(new_pos_x, new_pos_y)

        if (j + 1) % 2 != 0:
            dir_y *= -1
        else:
            dir_x *= -1
            dir_y *= -1

    # Move to start pos
    production_test_move(start_pos_x, start_pos_y)
    # Move max dist up for full speed
    production_test_move(end_pos_x, end_pos_y)
    # Move max dist down for full speed
    production_test_move(start_pos_x, start_pos_y)

    # Repeat the Test for the other diagonal

    # Move to start pos
    production_test_move(start_pos_x, end_pos_y)

    new_pos_x = start_pos_x
    new_pos_y = end_pos_y
    dir_x = +1
    dir_y = -1

    for j in range(4):
        for i in range(step_count):
            new_pos_x += 0.1 * dir_x
            new_pos_y += 0.1 * dir_y
            production_test_move(new_pos_x, new_pos_y)

            new_pos_x += 0.9 * dir_x
            new_pos_y += 0.9 * dir_y
            production_test_move(new_pos_x, new_pos_y)

            new_pos_x += (step_size_x - 1.0) * dir_x
            new_pos_y += (step_size_y - 1.0) * dir_y
            production_test_move(new_pos_x, new_pos_y)

        if (j + 1) % 2 != 0:
            dir_y *= -1
        else:
            dir_x *= -1
            dir_y *= -1

    # Move to start pos
    production_test_move(start_pos_x, end_pos_y)
    # Move max dist up for full speed
    production_test_move(end_pos_x, start_pos_y)
    # Move max dist down for full speed
    production_test_move(start_pos_x, end_pos_y)

    # Move to start pos
    production_test_move(start_pos_x, start_pos_y)


def production_test_move(position_x, position_y):
    global settling_time_x
    global settling_time_y
    global move_count_x
    global move_count_y

    result = move(position_x + ORIGIN_OFFSET_X, position_y + ORIGIN_OFFSET_Y)
    if 'settling_time_x' in result:
        settling_time_x[move_count_x] = result['settling_time_x']
        move_count_x += 1
    if 'settling_time_y' in result:
        settling_time_y[move_count_y] = result['settling_time_y']
        move_count_y += 1


def run_friction_scan(fixed_pos_x, fixed_pos_y, file_prefix=''):
    SCAN_SPEED = 10.0  # [mm/s]

    print_output(f"Starting Friction Scan")

    if elmo_axis_x is None or elmo_axis_y is None:
        print_output(f"X-axis and/or Y-axis not found", colors='red')
        return

    if axis_parameter is None:
        print_output(f"Scan table is not initialized", colors='red')
        return

    update_progress(0)

    friction_scan_datetime = datetime.now()
    friction_scan_results = {
        'date': friction_scan_datetime.strftime('%Y-%m-%d %H:%M:%S'),
        'fixed_pos_x': fixed_pos_x,
        'fixed_pos_y': fixed_pos_y,
        'speed_x': axis_parameter['SPEED_X'],
        'accel_x': axis_parameter['ACCEL_X'],
        'decel_x': axis_parameter['DECEL_X'],
        'speed_y': axis_parameter['SPEED_Y'],
        'accel_y': axis_parameter['ACCEL_Y'],
        'decel_y': axis_parameter['DECEL_Y'],
        'friction_scan_x': {},
        'friction_scan_y': {},
    }

    try:
        # Move to fixed position
        move(fixed_pos_x, fixed_pos_y)

        # Scan X axis first in positive direction, then in negative direction

        move_x(LOWER_LIMIT_X)

        error = elmo_axis_x.send_command(f"SP={round(SCAN_SPEED / ENCODER_SCALING)}")
        if error:
            raise ElmoAxisError("Failed to set X-axis speed")

        result = move_x(UPPER_LIMIT_X, current_scan_enable=True)
        friction_scan_results['friction_scan_x']['positive'] = {
            position: abs(current * MOTOR_FORCE_CONSTANT_X) for position, current in result['current_scan_x'].items()
        }
        update_progress(25)

        result = move_x(LOWER_LIMIT_X, current_scan_enable=True)
        friction_scan_results['friction_scan_x']['negative'] = {
            position: abs(current * MOTOR_FORCE_CONSTANT_X) for position, current in result['current_scan_x'].items()
        }
        update_progress(50)

        error = elmo_axis_x.send_command(f"SP={round(axis_parameter['SPEED_X'] / ENCODER_SCALING)}")
        if error:
            raise ElmoAxisError("Failed to set X-axis speed")

        # Return to fixed position
        move_x(fixed_pos_x)

        # Scan Y axis first in positive direction, then in negative direction

        move_y(LOWER_LIMIT_Y)

        error = elmo_axis_y.send_command(f"SP={round(SCAN_SPEED / ENCODER_SCALING)}")
        if error:
            raise ElmoAxisError("Failed to set Y-axis speed")

        result = move_y(UPPER_LIMIT_Y, current_scan_enable=True)
        friction_scan_results['friction_scan_y']['positive'] = {
            position: abs(current * MOTOR_FORCE_CONSTANT_Y) for position, current in result['current_scan_y'].items()
        }
        update_progress(75)

        result = move_y(LOWER_LIMIT_Y, current_scan_enable=True)
        friction_scan_results['friction_scan_y']['negative'] = {
            position: abs(current * MOTOR_FORCE_CONSTANT_Y) for position, current in result['current_scan_y'].items()
        }
        update_progress(100)

        error = elmo_axis_y.send_command(f"SP={round(axis_parameter['SPEED_Y'] / ENCODER_SCALING)}")
        if error:
            raise ElmoAxisError("Failed to set Y-axis speed")

        # Return to fixed position
        move_y(fixed_pos_y)

    except ElmoAxisError as e:
        print_output(f"Friction Scan aborted: {e}", colors='red')
        return

    file_name = f"friction_scan_{friction_scan_datetime.strftime('%Y%m%d_%H%M%S')}"
    if file_prefix != '':
        file_name = f"{file_prefix}_{file_name}"

    save_friction_scan_results(friction_scan_results, file_name)
    main_window.write_event_value('-FSCANRESULT-', (friction_scan_results, file_name))


def save_friction_scan_results(results, file_name):
    with open(os.path.join(FRICTION_SCAN_RESULTS_DIR, f"{file_name}.json"), 'w') as file:
        json.dump(results, file, indent=4)

    print_output(f"Friction Scan Results saved as {file_name}.json", colors='green')


def load_friction_scans():
    file_path = sg.popup_get_file('Select Multiple Friction Scan Result Files', no_window=True, multiple_files=True, file_types=(('JSON Files', '*.json'),))
    if not file_path:
        return

    friction_scan_list = []
    label_list = []
    for single_file_path in file_path:
        try:
            with open(single_file_path, 'r') as file:
                friction_scan = json.load(file)
                friction_scan['friction_scan_x']['positive'] = {
                    round(float(position), 6): friction for position, friction in friction_scan['friction_scan_x']['positive'].items()
                }
                friction_scan['friction_scan_x']['negative'] = {
                    round(float(position), 6): friction for position, friction in friction_scan['friction_scan_x']['negative'].items()
                }
                friction_scan['friction_scan_y']['positive'] = {
                    round(float(position), 6): friction for position, friction in friction_scan['friction_scan_y']['positive'].items()
                }
                friction_scan['friction_scan_y']['negative'] = {
                    round(float(position), 6): friction for position, friction in friction_scan['friction_scan_y']['negative'].items()
                }
                friction_scan_list.append(friction_scan)
                lable = os.path.splitext(os.path.basename(single_file_path))[0]
                lable = lable.split('_friction_scan_')[0]
                label_list.append(lable)
                print_output(f"Friction Scan Result loaded from {single_file_path}", colors='green')
        except Exception as e:
            print_output(f"Failed to load Friction Scan Result from {single_file_path}: {e}", colors='red')
            return

    show_friction_scan(friction_scan_list, label_list=label_list)


def show_friction_scan(friction_scan_list, label_list=[], file_name=None):
    if not isinstance(friction_scan_list, list):
        friction_scan_list = [friction_scan_list]
    if not isinstance(label_list, list):
        label_list = [label_list]

    if len(label_list) < len(friction_scan_list):
        label_list += [''] * (len(friction_scan_list) - len(label_list))

    figure_title = f"Friction Scan X-axis"
    plt.close(figure_title)
    plt.figure(figure_title)

    plt.subplot(2, 1, 1)
    for friction_scan, label in zip(friction_scan_list, label_list):
        positions = list(friction_scan['friction_scan_x']['positive'].keys())
        forces = list(friction_scan['friction_scan_x']['positive'].values())
        plt.plot(positions, forces, label=label)
    plt.axvline(x=PLATE_FIRST_COLUMN_X, color='red', linestyle='--', label='Plate Area')
    plt.axvline(x=PLATE_LAST_COLUMN_X, color='red', linestyle='--')
    plt.title(f"X-axis - Positive Direction")
    plt.xlabel('Position [mm]')
    plt.ylabel('Force [N]')
    plt.grid()
    plt.legend()

    plt.subplot(2, 1, 2)
    for friction_scan, label in zip(friction_scan_list, label_list):
        positions = list(friction_scan['friction_scan_x']['negative'].keys())
        forces = list(friction_scan['friction_scan_x']['negative'].values())
        plt.plot(positions, forces, label=label)
    plt.axvline(x=PLATE_FIRST_COLUMN_X, color='red', linestyle='--', label='Plate Area')
    plt.axvline(x=PLATE_LAST_COLUMN_X, color='red', linestyle='--')
    plt.title(f"X-axis - Negative Direction")
    plt.xlabel('Position [mm]')
    plt.ylabel('Force [N]')
    plt.grid()
    plt.legend()

    plt.tight_layout(h_pad=2.0)

    if file_name is not None:
        file_name_x = file_name.replace('friction_scan', 'friction_scan_x_axis')
        plt.savefig(os.path.join(FRICTION_SCAN_RESULTS_DIR, f"{file_name_x}.png"))
        print_output(f"Friction Scan X-axis Results saved as {file_name_x}.png", colors='green')

    plt.show(block=False)

    figure_title = f"Friction Scan Y-axis"
    plt.close(figure_title)
    plt.figure(figure_title)

    plt.subplot(2, 1, 1)
    for friction_scan, label in zip(friction_scan_list, label_list):
        positions = list(friction_scan['friction_scan_y']['positive'].keys())
        forces = list(friction_scan['friction_scan_y']['positive'].values())
        plt.plot(positions, forces, label=label)
    plt.axvline(x=PLATE_FIRST_ROW_Y, color='red', linestyle='--', label='Plate Area')
    plt.axvline(x=PLATE_LAST_ROW_Y, color='red', linestyle='--')
    plt.title(f"Y-axis - Positive Direction")
    plt.xlabel('Position [mm]')
    plt.ylabel('Force [N]')
    plt.grid()
    plt.legend()

    plt.subplot(2, 1, 2)
    for friction_scan, label in zip(friction_scan_list, label_list):
        positions = list(friction_scan['friction_scan_y']['negative'].keys())
        forces = list(friction_scan['friction_scan_y']['negative'].values())
        plt.plot(positions, forces, label=label)
    plt.axvline(x=PLATE_FIRST_ROW_Y, color='red', linestyle='--', label='Plate Area')
    plt.axvline(x=PLATE_LAST_ROW_Y, color='red', linestyle='--')
    plt.title(f"Y-axis - Negative Direction")
    plt.xlabel('Position [mm]')
    plt.ylabel('Force [N]')
    plt.grid()
    plt.legend()

    plt.tight_layout(h_pad=2.0)

    if file_name is not None:
        file_name_y = file_name.replace('friction_scan', 'friction_scan_x_axis')
        plt.savefig(os.path.join(FRICTION_SCAN_RESULTS_DIR, f"{file_name_y}.png"))
        print_output(f"Friction Scan Y-axis Results saved as {file_name_y}.png", colors='green')

    plt.show(block=False)


if __name__ == '__main__':
    main()
